# dynamodb

Introduction to Amazon DynamoDB

## Technology

[Amazon DynamoDB](https://aws.amazon.com/dynamodb/), [Amazon DynamoDB NoSQL Workbench](https://aws.amazon.com/dynamodb/nosql-workbench/), and [PartiQL](https://partiql.org/).

## Objective

Create and query an Amazon DynamoDB database.

## References

- [Amazon DynamoDB Developer Guide](https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/Introduction.html)

## Development

### Working with Your AWS Builder ID

Create and use your [AWS Builder ID](https://docs.aws.amazon.com/signin/latest/userguide/sign-in-aws_builder_id.html)

### Create an AWS IAM User

In AWS IAM, create a **DynamoDBDeveloper** user with the following attached policies.

- **AmazonDynamoDBFullAccess**

Create an access key for the **DynamoDBDeveloper** user.

Create a **LambdaDeveloper** profile with **aws configure --profile DynamoDBDeveloper**. Activate the profile by appending it to **aws** or **sam** commands with **--profile**.

### Define Table with the AWS CLI

Run the [create music table scripts](./create-music-table.bat). Note the [DynamoDB naming rules and data types](https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/HowItWorks.NamingRulesDataTypes.html).

### Add Item with the AWS Console

Add the following DynamoDB JSON. DynamoDB does not use the classical JSON format to store items internally. Instead, it uses a "marshalled" format. DynamoDB wraps all the attribute values in objects where the Key indicates its type and attribute value stays as-is. For instance: **{ value: 3 }** becomes **{ value: { N: "3" } }**.

```json
{
  "Artist": {
    "S": "The Beatles"
  },
  "Song": {
    "S": "1969-Something"
  },
  "Duration": {
    "S": "03:03"
  },
  "Album": {
    "S": "1"
  }
}
```

### Add Item with the NoSQL Workbench

Open the [Amazon DynamoDB NoSQL Workbench](https://aws.amazon.com/dynamodb/nosql-workbench/). The tab _Operation builder_ lists active connections based on the local user profiles (i.e. _DynamoDBDeveloper_).

Select region _eu-west-1_ and table _Music_.

Use an _interface-based operation_ to add the following song.

```bash
{
  "Artist": {
    "S": "The Beatles"
  },
  "Song": {
    "S": "1968-Hey Jude"
  },
  "Duration": {
    "S": "07:08"
  },
  "Album": {
    "S": "1"
  }
}
```

Generate JavaScript (Node.js) code and copy it into [index.js](./index.js).

### Add Items with JavaScript (Node.js)

Install the AWS SDK.

```bash
npm i aws-sdk
```

Configure the AWS SDK to use the _DynamoDBDeveloper_ profile.

```js
var credentials = new AWS.SharedIniFileCredentials({ profile: "work-account" });
AWS.config.credentials = credentials;
```

### Query Table with PartiQL

Run the following queries in the [Amazon DynamoDB NoSQL Workbench](https://aws.amazon.com/dynamodb/nosql-workbench/).

```sql
select * from Music where Artist = 'The Beatles'

select * from Music where Song between '1965' and '1971'

select * from Music where Duration >= '05' -- Full scan required

select * from Music where Album = '1' order by Duration -- Error: Attribute Duration in ORDER BY clause must be part of the primary key
```

### Create a Local Secondary Index

Delete table **Music** and [recreate it with a Local Secondary Index](./create-music-table-with-lsi.bat).

In the [Amazon DynamoDB NoSQL Workbench](https://aws.amazon.com/dynamodb/nosql-workbench/), run the following [PartiQL](https://partiql.org/) statements to add songs.

```sql
INSERT INTO "Music" value {'Artist' : 'The Beatles', 'Song' : '1969-Something', 'Duration' : '03:03', 'Album' : '1'}

INSERT INTO "Music" value {'Artist' : 'The Beatles', 'Song' : '1968-Hey Jude', 'Duration' : '07:08', 'Album' : '1'}

INSERT INTO "Music" value {'Artist' : 'The Beatles', 'Song' : '1964-She Loves You', 'Duration' : '02:21', 'Album' : '1'}
```

With an LSI, following queries will be feasible and efficient...

```sql
select * from Music.MusicLSI where Artist = 'The Beatles' order by Duration

select * from Music.MusicLSI where Duration >= '05' -- Query instead of scan
```

... and the following queries not.

```sql
select * from Music where Album = '1' -- Scan for album

select * from Music where Album = '1' and Song between '1960' and '1965' -- Scan for album
```

### Create a Global Secondary Index

In the AWS Console, create a Global Secondary Index with partition Key _Album_ and sort key _Song_.

With an LSI, the following queries will be efficient, too.

```sql
select * from Music where Album = '1' -- Scan for album

select * from Music where Album = '1' and Song between '1960' and '1965' -- Scan for album
```
