const chalk = require("chalk");
const express = require("express");
const { body, validationResult } = require("express-validator");
const Project = require("../../models/Project");
const Client = require("../../models/Client");

const router = express.Router();

/**
 * @route   POST api/projects/
 * @desc    Create a project
 * @access  Public
 */
router.post(
  "/",
  [
    body("client", "Client is required").not().isEmpty(),
    body("name", "Name is required").not().isEmpty(),
    body("status", "Email is required").not().isEmpty(),
    body("description", "Phone is required").not().isEmpty(),
  ],
  async (req, res) => {
    try {
      const errors = validationResult(req);

      if (!errors.isEmpty()) {
        return res.status(400).json({ errors: errors.array() });
      }

      const { client: clientId, name, status, description } = req.body;

      const client = await Client.findById({
        _id: clientId,
      });

      if (!client) {
        return res.status(400).json({ errors: [{ msg: "No client" }] });
      }

      const project = new Project({
        client: client._id,
        name,
        status,
        description,
      });

      await project.save();

      res.json(project);
    } catch (error) {
      console.error(chalk.red(error.message));

      res.status(500).json({ errors: [{ msg: "Server error" }] });
    }
  }
);

/**
 * @route   PUT api/projects/:id
 * @desc    Update the project with the given id
 * @access  Public
 */
router.put("/:id", async (req, res) => {
  try {
    const id = req.params.id;

    if (!id) {
      return res.status(400).json({ errors: [{ msg: "No project id" }] });
    }

    let { client: clientId, name, status, description } = req.body;

    if (clientId) {
      const client = await Client.findById({
        _id: clientId,
      });

      if (!client) {
        return res.status(400).json({ errors: [{ msg: "No client" }] });
      }
    }

    const projectFields = {
      clientId,
      name,
      status,
      description,
    };

    const updatedProject = await Project.findOneAndUpdate(
      { _id: req.params.id },
      { $set: projectFields },
      { new: true, upsert: false }
    );

    res.json(updatedProject);
  } catch (error) {
    console.error(chalk.red(error.message));

    if (error.kind === "ObjectId") {
      return res.status(400).json({ errors: [{ msg: "No client" }] });
    }

    res.status(500).json({ errors: [{ msg: "Server error" }] });
  }
});

/**
 * @route   GET api/projects
 * @desc    Get all projects
 * @access  Public
 */
router.get("/", async (req, res) => {
  try {
    const projects = await Project.find();

    res.json(projects);
  } catch (error) {
    console.error(chalk.red(error.message));
    res.status(500).json({ errors: [{ msg: "Server error" }] });
  }
});

/**
 * @route   GET api/projects/:id
 * @desc    Get the project with the given id
 * @access  Public
 */
router.get("/:id", async (req, res) => {
  try {
    const project = await Project.findById(req.params.id);

    res.json(project);
  } catch (error) {
    console.error(chalk.red(error.message));
    res.status(500).json({ errors: [{ msg: "Server error" }] });
  }
});

/**
 * @route   DELETE api/projects/:id
 * @desc    Delete the project with the given id
 * @access  Public
 */
router.delete("/:id", async (req, res) => {
  try {
    await Project.findByIdAndRemove(req.params.id);

    res.json({ msg: "Project deleted" });
  } catch (error) {
    console.error(chalk.red(error.message));
    res.status(500).json({ errors: [{ msg: "Server error" }] });
  }
});

module.exports = router;
