const chalk = require("chalk");
const express = require("express");
const { body, validationResult } = require("express-validator");
const Client = require("../../models/Client");

const router = express.Router();

/**
 * @route   POST api/clients/
 * @desc    Create a client
 * @access  Public
 */
router.post(
  "/",
  [
    body("name", "Name is required").not().isEmpty(),
    body("email", "Email is required").not().isEmpty(),
    body("phone", "Phone is required").not().isEmpty(),
  ],
  async (req, res) => {
    try {
      const errors = validationResult(req);

      if (!errors.isEmpty()) {
        return res.status(400).json({ errors: errors.array() });
      }

      let { name, email, phone } = req.body;

      const client = new Client({
        name,
        email,
        phone,
      });

      await client.save();

      res.json(client);
    } catch (error) {
      console.error(chalk.red(error.message));

      res.status(500).json({ errors: [{ msg: "Server error" }] });
    }
  }
);

/**
 * @route   PUT api/clients/:id
 * @desc    Update the client with the given id
 * @access  Public
 */
router.put("/:id", async (req, res) => {
  try {
    const id = req.params.id;

    if (!id) {
      return res.status(400).json({ errors: [{ msg: "No client id" }] });
    }

    let { name, email, phone } = req.body;

    const clientFields = {
      name,
      email,
      phone,
    };

    const updatedClient = await Client.findOneAndUpdate(
      { _id: req.params.id },
      { $set: clientFields },
      { new: true, upsert: false }
    );

    res.json(updatedClient);
  } catch (error) {
    console.error(chalk.red(error.message));

    if (error.kind === "ObjectId") {
      return res.status(400).json({ errors: [{ msg: "No client" }] });
    }

    res.status(500).json({ errors: [{ msg: "Server error" }] });
  }
});

/**
 * @route   GET api/clients
 * @desc    Get all clients
 * @access  Public
 */
router.get("/", async (req, res) => {
  try {
    const clients = await Client.find();

    res.json(clients);
  } catch (error) {
    console.error(chalk.red(error.message));
    res.status(500).json({ errors: [{ msg: "Server error" }] });
  }
});

/**
 * @route   GET api/clients/:id
 * @desc    Get the client with the given id
 * @access  Public
 */
router.get("/:id", async (req, res) => {
  try {
    const client = await Client.findById(req.params.id);

    res.json(client);
  } catch (error) {
    console.error(chalk.red(error.message));
    res.status(500).json({ errors: [{ msg: "Server error" }] });
  }
});

/**
 * @route   DELETE api/clients/:id
 * @desc    Delete the client with the given id
 * @access  Public
 */
router.delete("/:id", async (req, res) => {
  try {
    Project.find({ client: req.params.id }).then((projects) => {
      projects.forEach((project) => {
        project.deleteOne();
      });
    });

    await Client.findByIdAndRemove(req.params.id);

    res.json({ msg: "Client deleted" });
  } catch (error) {
    console.error(chalk.red(error.message));
    res.status(500).json({ errors: [{ msg: "Server error" }] });
  }
});

module.exports = router;
